/**
 * \file keyman.c
 *
 * \brief Architecture independent keyman functions.
 *
 * These functions will internally call architecture dependent functions for
 * realization
 *
 * \author Christoph Gellner (cgellner@de.adit-jv.com)
 *
 * \copyright (c) 2015 Advanced Driver Information Technology.
 * This code is developed by Advanced Driver Information Technology.
 * Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
 * All rights reserved.
 *
 *
 ***********************************************************************/


#include <stdlib.h>
#include <string.h>
#include <stdio.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>

#include <keyman.h>
#include <private/keyman_arch.h>
#include <private/keyman_intern.h>
#include <sdc/daemon/arch/sdc_daemon.h>

#define KEYMAN_RANDOM_DEVICE "/dev/urandom"
#define VERSION_CHECK_UNAVAIL_ERROR "libKeyman version check not active"

/* libkeyman external */
sdc_error_t keyman_kernel_version_verify(char **version_info_str)
{
    sdc_error_t err = keyman_arch_kernel_version_verify(version_info_str);

    if (err == SDC_NOT_SUPPORTED) {
        err = SDC_OK;

        if (version_info_str != NULL) {
            *version_info_str = malloc(strlen(VERSION_CHECK_UNAVAIL_ERROR)+1);

            if (*version_info_str) {
                sprintf(*version_info_str,
                        VERSION_CHECK_UNAVAIL_ERROR);
            } else {
                err = SDC_NO_MEM;
            }
        }
    }

    return err;
}

sdc_error_t keyman_ctx_alloc(libkeyman_ctx_t **ctx)
{
    if (ctx == NULL)
        return SDC_INVALID_PARAMETER;

    return keyman_arch_ctx_alloc(ctx);
}

sdc_error_t keyman_ctx_free(libkeyman_ctx_t *ctx)
{
    if (ctx == NULL)
        return SDC_INVALID_PARAMETER;

    return keyman_arch_ctx_free(ctx);
}


sdc_error_t keyman_load_persistent_keystore(libkeyman_ctx_t *ctx, int fd,
                                            size_t *loaded_keys, size_t *existing_keys)
{
    sdc_error_t err;
    size_t loaded_keys_local;
    size_t existing_keys_local;

    if (!ctx)
        return SDC_INVALID_PARAMETER;

    if (fd < 0)
        return SDC_INVALID_PARAMETER;

    err = keyman_arch_load_persistent_keystore(ctx, fd,
                                               &loaded_keys_local, &existing_keys_local);

    if (loaded_keys) {
        *loaded_keys = loaded_keys_local;
    }
    if (existing_keys) {
        *existing_keys = existing_keys_local;
    }

    return err;
}

sdc_error_t keyman_store_persistent_keystore(libkeyman_ctx_t *ctx, int fd, size_t *stored_keys)
{
    sdc_error_t err;
    size_t stored_keys_local;

    if (!ctx)
        return SDC_INVALID_PARAMETER;

    if (fd < 0)
        return SDC_INVALID_PARAMETER;

    err = keyman_arch_store_persistent_keystore(ctx, fd, &stored_keys_local);

    if (stored_keys) {
        *stored_keys = stored_keys_local;
    }

    return err;
}

sdc_error_t keyman_empty_persistent_keystore(libkeyman_ctx_t *ctx, int fd)
{
    if (!ctx)
        return SDC_INVALID_PARAMETER;

    if (fd < 0)
        return SDC_INVALID_PARAMETER;

    return keyman_arch_empty_persistent_keystore(ctx, fd);
}

sdc_error_t keyman_flush_keystore(libkeyman_ctx_t *ctx,
                                  libkeyman_keystore_cont_t keystore_content, bool force)
{
    if (!ctx)
        return SDC_INVALID_PARAMETER;

    /* currently limited to complete keystore */
    if (keystore_content != LIBKEYMAN_KEYSTORE_ALL)
        return SDC_INVALID_PARAMETER;

    return keyman_arch_flush_keystore(ctx, keystore_content, force);
}

sdc_error_t keyman_lock_unlock_keystore(libkeyman_ctx_t *ctx,
                                        libkeyman_keystore_cont_t keystore_content, bool lock)
{
    if (!ctx)
        return SDC_INVALID_PARAMETER;

    /* currently limited to complete keystore */
    if (keystore_content != LIBKEYMAN_KEYSTORE_ALL)
        return SDC_INVALID_PARAMETER;

    return keyman_arch_lock_unlock_keystore(ctx, keystore_content, lock);
}

/* key_spec needs to be checked before calling this function */
static sdc_error_t keyman_generate_common_check_secret(
    const libkeyman_key_spec_t *key_spec,
    const uint8_t *key_data,
    size_t key_data_len)
{
    sdc_error_t err = SDC_OK;
    size_t exp_secret_len;

    if ((!key_data) || (key_data_len == 0)) {
        err = SDC_INVALID_PARAMETER;
    }

    if ((err == SDC_OK) && (key_spec->fmt == SDC_KEY_FMT_SIMPLE_SYM)) {
        err = sdc_key_len_get_bytes(key_spec->len, &exp_secret_len);
        if ((err == SDC_OK) && (key_data_len != exp_secret_len)) {
            err = SDC_INVALID_PARAMETER;
        }
    }

    return err;
}

static sdc_error_t keyman_generate_common_check_inputs_explicit_kid(
    libkeyman_ctx_t *ctx,
    sdc_key_id_t key_id,
    const libkeyman_key_spec_t *key_spec)
{
    if (!ctx)
        return SDC_INVALID_PARAMETER;

    if (key_id == SDC_FLAG_INVALID_KID)
        return SDC_KEY_INVALID;

    if (!key_spec)
        return SDC_INVALID_PARAMETER;

    if (!key_spec->permissions)
        return SDC_INVALID_PARAMETER;

    if ((key_spec->len < SDC_KEY_LEN_FIRST) || (key_spec->len >= SDC_KEY_LEN_END))
        return SDC_KEYLEN_INVALID;

    if ((key_spec->fmt < SDC_KEY_FMT_FIRST) || (key_spec->fmt >= SDC_KEY_FMT_END))
        return SDC_KEY_FMT_INVALID;

    return sdc_daemon_check_permission_range(key_spec->permissions);
}

static sdc_error_t keyman_generate_common_check_inputs_automatic_kid(
    libkeyman_ctx_t *ctx,
    const sdc_key_id_range_t *key_id_range,
    const libkeyman_key_spec_t *key_spec)
{
    sdc_error_t err;

    err = keyman_generate_common_check_inputs_explicit_kid(
        ctx,
        key_id_range->first,
        key_spec);

    if (err == SDC_OK) {
        if (!key_id_range->result)
            err = SDC_INVALID_PARAMETER;

        if (key_id_range->last == SDC_FLAG_INVALID_KID)
            err = SDC_KEY_INVALID;

        if (key_id_range->last < key_id_range->first)
            err = SDC_KEY_INVALID;
    }

    return err;
}

sdc_error_t keyman_generate_plain_storage_key_explicit_kid(
    libkeyman_ctx_t *ctx,
    sdc_key_id_t key_id,
    const libkeyman_key_spec_t *key_spec,
    bool persistent,
    bool activate_immediately,
    sdc_keysecret_enc_t key_data_enc,
    const uint8_t *key_data,
    size_t key_data_len)
{
    sdc_error_t err;
    sdc_key_id_range_t key_id_range = {
            .first = key_id,
            .last = key_id,
            .result = NULL,
    };
    sdc_keysecret_t key_secret = {
            .enc = key_data_enc,
            .secret = key_data,
            .secret_len = key_data_len,
    };

    err = keyman_generate_common_check_inputs_explicit_kid(
        ctx,
        key_id,
        key_spec);

    if (err == SDC_OK)
        err = keyman_generate_common_check_secret(
            key_spec,
            key_data, key_data_len);

    if (err == SDC_OK) {
        err = keyman_arch_generate_storage_key(
            ctx,
            &key_id_range,
            key_spec,
            persistent, activate_immediately,
            &key_secret);
    }

    return err;
}

sdc_error_t keyman_generate_plain_storage_key_automatic_kid(
    libkeyman_ctx_t *ctx,
    sdc_key_id_t key_id_first,
    sdc_key_id_t key_id_last,
    sdc_key_id_t *result_key_id,
    const libkeyman_key_spec_t *key_spec,
    bool persistent,
    bool activate_immediately,
    sdc_keysecret_enc_t key_data_enc,
    const uint8_t *key_data,
    size_t key_data_len)
{
    sdc_error_t err;
    sdc_key_id_range_t key_id_range = {
            .first = key_id_first,
            .last = key_id_last,
            .result = result_key_id,
    };
    sdc_keysecret_t key_secret = {
            .enc = key_data_enc,
            .secret = key_data,
            .secret_len = key_data_len,
    };

    err = keyman_generate_common_check_inputs_automatic_kid(
        ctx,
        &key_id_range,
        key_spec);

    if (err == SDC_OK)
        err = keyman_generate_common_check_secret(
            key_spec,
            key_data, key_data_len);

    if (err == SDC_OK) {
        err = keyman_arch_generate_storage_key(
            ctx,
            &key_id_range,
            key_spec,
            persistent, activate_immediately,
            &key_secret);
    }

    return err;
}

sdc_error_t keyman_generate_random_storage_key_explicit_kid(
    libkeyman_ctx_t *ctx,
    sdc_key_id_t key_id,
    const libkeyman_key_spec_t *key_spec,
    bool persistent,
    bool activate_immediately)
{
    sdc_error_t err;
    sdc_key_id_range_t key_id_range = {
            .first = key_id,
            .last = key_id,
            .result = NULL,
    };

    err = keyman_generate_common_check_inputs_explicit_kid(
        ctx,
        key_id,
        key_spec);

    if (err == SDC_OK) {
        err = keyman_arch_generate_storage_key(
            ctx, &key_id_range,
            key_spec,
            persistent, activate_immediately,
            NULL);
    }

    return err;
}

sdc_error_t keyman_generate_random_storage_key_automatic_kid(
    libkeyman_ctx_t *ctx,
    sdc_key_id_t key_id_first,
    sdc_key_id_t key_id_last,
    sdc_key_id_t *result_key_id,
    const libkeyman_key_spec_t *key_spec,
    bool persistent,
    bool activate_immediately)
{
    sdc_error_t err;
    sdc_key_id_range_t key_id_range = {
            .first = key_id_first,
            .last = key_id_last,
            .result = result_key_id,
    };

    err = keyman_generate_common_check_inputs_automatic_kid(
        ctx,
        &key_id_range,
        key_spec);

    if (err == SDC_OK) {
        err = keyman_arch_generate_storage_key(
            ctx,
            &key_id_range,
            key_spec,
            persistent, activate_immediately,
            NULL);
    }

    return err;
}

sdc_error_t keyman_import_storage_key_explicit_kid(
    libkeyman_ctx_t *ctx,
    sdc_key_id_t key_id,
    const libkeyman_key_spec_t *key_spec,
    bool persistent,
    bool activate_immediately,
    sdc_keysecret_enc_t key_data_enc,
    const uint8_t *key_data,
    size_t key_data_len)
{
    sdc_error_t err;
    sdc_key_id_range_t key_id_range = {
            .first = key_id,
            .last = key_id,
            .result = NULL,
    };
    sdc_keysecret_t key_secret = {
            .enc = key_data_enc,
            .secret = key_data,
            .secret_len = key_data_len,
    };

    err = keyman_generate_common_check_inputs_explicit_kid(
        ctx,
        key_id,
        key_spec);

    if (err == SDC_OK)
        if ((!key_data) || (key_data_len == 0)) {
            err = SDC_INVALID_PARAMETER;
        }

    if (err == SDC_OK) {
        err = keyman_arch_import_storage_key(
            ctx,
            &key_id_range,
            key_spec,
            persistent, activate_immediately,
            &key_secret);
    }

    return err;
}

sdc_error_t keyman_import_storage_key_automatic_kid(
    libkeyman_ctx_t *ctx,
    sdc_key_id_t key_id_first,
    sdc_key_id_t key_id_last,
    sdc_key_id_t *result_key_id,
    const libkeyman_key_spec_t *key_spec,
    bool persistent,
    bool activate_immediately,
    sdc_keysecret_enc_t key_data_enc,
    const uint8_t *key_data,
    size_t key_data_len)
{
    sdc_error_t err;
    sdc_key_id_range_t key_id_range = {
            .first = key_id_first,
            .last = key_id_last,
            .result = result_key_id,
    };
    sdc_keysecret_t key_secret = {
            .enc = key_data_enc,
            .secret = key_data,
            .secret_len = key_data_len,
    };

    err = keyman_generate_common_check_inputs_automatic_kid(
        ctx,
        &key_id_range,
        key_spec);

    if (err == SDC_OK)
        if ((!key_data) || (key_data_len == 0)) {
            err = SDC_INVALID_PARAMETER;
        }

    if (err == SDC_OK) {
        err = keyman_arch_import_storage_key(
            ctx,
            &key_id_range,
            key_spec,
            persistent, activate_immediately,
            &key_secret);
    }

    return err;
}


sdc_error_t keyman_import_insecure_product_key(
    libkeyman_ctx_t *ctx,
    sdc_key_id_t key_id,
    const libkeyman_key_spec_t *key_spec,
    bool activate_immediately,
    sdc_keysecret_enc_t key_data_enc,
    const uint8_t *key_data,
    size_t key_data_len)
{
    sdc_error_t err;
    sdc_keysecret_t key_secret = {
            .enc = key_data_enc,
            .secret = key_data,
            .secret_len = key_data_len,
    };

    err = keyman_generate_common_check_inputs_explicit_kid(
        ctx,
        key_id,
        key_spec);

    if (err == SDC_OK)
        if ((!key_data) || (key_data_len == 0)) {
            err = SDC_INVALID_PARAMETER;
        }

    if (err == SDC_OK) {
        err = keyman_arch_import_insecure_product_key(
            ctx,
            key_id,
            key_spec,
            activate_immediately,
            &key_secret);
    }

    return err;
}

sdc_error_t keyman_activate_key(
    libkeyman_ctx_t *ctx,
    sdc_key_id_t key_id)
{
    if (!ctx)
        return SDC_INVALID_PARAMETER;

    if (key_id == SDC_FLAG_INVALID_KID)
        return SDC_KEY_INVALID;

    return keyman_arch_activate_key(ctx, key_id);
}

sdc_error_t keyman_remove_storage_key(
    libkeyman_ctx_t *ctx,
    sdc_key_id_t key_id,
    uid_t uid, gid_t gid,
    bool *result_is_persistent
    )
{
    sdc_error_t err;
    bool persistent;

    if (!ctx)
        return SDC_INVALID_PARAMETER;

    if (key_id == SDC_FLAG_INVALID_KID)
        return SDC_KEY_INVALID;

    err = keyman_arch_remove_storage_key(ctx, key_id, uid, gid, &persistent);

    if ((err == SDC_OK) && (result_is_persistent)) {
        *result_is_persistent = persistent;
    }

    return err;
}


/* libkeyman internal functions */
sdc_error_t keyman_intern_search_kid_init(sdc_key_id_t first,
                                          sdc_key_id_t last, sdc_key_id_t *init_kid)
{
    sdc_error_t err = SDC_OK;
    sdc_key_id_t kids;
    sdc_key_id_t rand_kid;
    int fd;
    ssize_t read_bytes;

    if (first == last) {
        *init_kid = first;
    } else {
        fd = open(KEYMAN_RANDOM_DEVICE, O_RDONLY);
        if (fd < 0)
            return SDC_UNKNOWN_ERROR;

        read_bytes = read(fd, &rand_kid, sizeof(rand_kid));
        if (read_bytes != sizeof(rand_kid))
            err = SDC_UNKNOWN_ERROR;

        if (err == SDC_OK) {
            kids = (last - first) + 1;
            rand_kid = rand_kid % kids;
            rand_kid += first;
            *init_kid = rand_kid;
        }

        close (fd);
    }

    return err;
}

sdc_error_t keyman_intern_search_kid_next(sdc_key_id_t first,
                                          sdc_key_id_t last, sdc_key_id_t init_kid, sdc_key_id_t *kid)
{
    sdc_error_t err = SDC_KID_EXISTS;

    if (first != last) {
        *kid += 1;

        if (*kid > last)
            *kid = first;

        if (*kid != init_kid)
            err = SDC_OK;
    }

    return err;
}
